<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class RecipeController extends Controller
{
    public function showForm()
    {
        return view('recipe_form');
    }

    public function generateRecipe(Request $request)
    {
        $request->validate([
            'ingredients' => 'nullable|string',
            'cuisine' => 'nullable|in:North Indian,South Indian,Gujarati,Bengali,Maharashtrian,Goan,Italian,Mexican,Chinese,Thai',
            'meal_type' => 'nullable|in:Breakfast,Lunch,Dinner,Snack,Dessert',
            'difficulty' => 'nullable|in:Beginner,Intermediate,Advanced',
            'cooking_time' => 'nullable|in:Under 30 minutes,30-60 minutes,60–90 minutes,Over 90 minutes',
            'dietary_restrictions' => 'nullable|array',
            'dietary_restrictions.*' => 'in:Vegetarian,Gluten-free,Diabetic-friendly,Low-calorie,High-protein',
            'servings' => 'nullable|integer|min:1',
            'veg_nonveg' => 'nullable|in:Veg,Non-Veg',
            'additional_instructions' => 'nullable|string',
        ]);

        $apiKey = env('API_KEY');
        $workflow = env('WORKFLOW_ID');

        $additional_instructions = $request->input('additional_instructions', '');
        $veg_nonveg = $request->input('veg_nonveg', '');
        if ($veg_nonveg) {
            $additional_instructions .= ($additional_instructions ? ', ' : '') . $veg_nonveg;
        }

        $data = [
            'workflow' => $workflow,
            'input' => [
                'ingredients' => array_map('trim', explode(',', $request->input('ingredients', ''))),
                'cuisine' => $request->input('cuisine', ''),
                'meal_type' => $request->input('meal_type', ''),
                'difficulty' => $request->input('difficulty', 'Beginner'),
                'cooking_time' => $request->input('cooking_time', 'Under 30 minutes'),
                'dietary_restrictions' => $request->input('dietary_restrictions', []),
                'servings' => (int) $request->input('servings', 1),
                'additional_instructions' => $additional_instructions,
            ],
        ];

        // Log the request data without the API key
        Log::info('Request Data:', $data);

        $response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'apikey' =>   $apiKey,
        ])->post('https://hub.peppercontent.io/workflows/v1/run', $data);

        // Log the response status and body
        Log::info('Response Status: ' . $response->status());
        Log::info('Response Body: ' . $response->body());

        if ($response->successful()) {
            $recipe = json_decode($response->json()['output'], true);
            return view('recipe_result', compact('recipe'));
        } else {
            $errorMessage = $response->json()['message'] ?? 'Failed to fetch the recipe. Please try again later.';
            return back()->withErrors(['msg' => $errorMessage]);
        }
    }
}
